<?php
/**
 * Helper functions for theme templates (Theme 2)
 */

function casino_b2_get_header_logo() {
    $lang_slug = pll_current_language();
    $field_name = 'header_logo_b2_' . $lang_slug;
    $logo = get_field($field_name, 'options_' . $lang_slug);
    return $logo ?: ['url' => get_template_directory_uri() . '/assets/public/img/logo.png', 'alt' => get_bloginfo('name')];
}

function casino_b2_get_header_menu() {
    $lang_slug = pll_current_language();
    $field_name = 'header_menu_b2_' . $lang_slug;
    $menu_items = get_field($field_name, 'options_' . $lang_slug);

    if (empty($menu_items)) {
        return [];
    }

    return array_map(function($item) {
        if (empty($item['url'])) {
            $item['url'] = '#';
        }
        if (!empty($item['submenu'])) {
            $item['submenu'] = array_map(function($sub_item) {
                if (empty($sub_item['url'])) {
                    $sub_item['url'] = '#';
                }
                return $sub_item;
            }, $item['submenu']);
        }
        return $item;
    }, $menu_items);
}


function casino_b2_get_footer_menu() {
    $lang_slug = pll_current_language();
    $field_name = 'footer_menu_b2_' . $lang_slug;
    return get_field($field_name, 'options_' . $lang_slug) ?: [];
}

function casino_b2_get_footer_payments() {
    $lang_slug = pll_current_language();
    $field_name = 'footer_payments_b2_' . $lang_slug;
    return get_field($field_name, 'options_' . $lang_slug) ?: [];
}

function casino_b2_get_footer_copyright_img() {
    $lang_slug = pll_current_language();
    $field_name = 'footer_copyright_img_b2_' . $lang_slug;
    return get_field($field_name, 'options_' . $lang_slug);
}

function casino_b2_get_footer_copyright_text() {
    $lang_slug = pll_current_language();
    $field_name = 'footer_copyright_text_b2_' . $lang_slug;
    $copyright_text = get_field($field_name, 'options_' . $lang_slug);

    if ($copyright_text) {
        return '© ' . date('Y') . ' ' . $copyright_text;
    }

    return '© ' . date('Y') . ' All rights reserved';
}

function casino_b2_get_global_button_url() {
    $lang_slug = pll_current_language();
    $field_name = 'global_button_url_b2_' . $lang_slug;
    $url = get_field($field_name, 'options_' . $lang_slug);
    
    if ($url) {
        $url = trim($url);

        if ($url === '') {
            return '#';
        }

        $parsed = wp_parse_url($url);
        if (!empty($parsed['scheme']) || str_starts_with($url, '//') || str_starts_with($url, '#')) {
            return $url;
        }

        if (str_starts_with($url, '/')) {
            return $url;
        }

        return '/' . ltrim($url, '/');
    }
    
    return '#';
}

/**
 * Extracts H2 tags from content to build a table of contents.
 */
function extract_h2_tags($content) {
    if (empty($content)) return [];
    preg_match_all('/<h2.*?>(.*?)<\/h2>/', $content, $matches);
    return $matches[1];
}

function add_ids_to_h2_tags($content) {
    $index = 0;
    return preg_replace_callback('/<h2.*?>(.*?)<\/h2>/', function($matches) use (&$index) {
        $slug = sanitize_title($matches[1]) . '-' . $index++;
        return '<h2 id="' . esc_attr($slug) . '">' . $matches[1] . '</h2>';
    }, $content);
}
add_filter('the_content', 'add_ids_to_h2_tags');

/**
 * Get static image attributes
 *
 * @param string $specific_alt Optional specific alt text for the image
 * @param string $image_name Optional image name (for title attribute only)
 * @param string $image_type Optional image type (e.g., 'flag' for language flags)
 * @param string $flag_lang Optional language code for flag (e.g., 'en', 'de')
 * @return array Array with alt and title attributes
 */
function get_static_image_attr($specific_alt = '', $image_name = '', $image_type = '', $flag_lang = '') {
    // Current site language
    $current_site_lang = pll_current_language() ?: 'en';
    $post_id = 'options_' . $current_site_lang;
    
    // Get global alt and title values for current site language
    $global_alt = get_field('static_images_alt_b2_' . $current_site_lang, $post_id) ?: get_bloginfo('name');
    $global_title = get_field('static_images_title_b2_' . $current_site_lang, $post_id) ?: get_bloginfo('name');
    
    // For flag images, use language-specific flag alt text from the admin fields
    if ($image_type === 'flag' && !empty($flag_lang)) {
        // Get language-specific flag alt text from the admin fields
        $flag_specific_alt = get_field('flag_' . $flag_lang . '_alt_b2_' . $current_site_lang, $post_id);
        
        if (!empty($flag_specific_alt)) {
            // Use the exact value from admin field for alt
            $global_alt = $flag_specific_alt;
        } else {
            // Fallback to language name if the field is empty
            $global_alt = strtoupper($flag_lang) . ' ' . __('language', 'base-theme-2');
        }
        // For flags, the title should be the same as the alt text (language name)
        $global_title = $global_alt;
    }
    
    // Use specific alt if provided, otherwise use global
    $alt = !empty($specific_alt) ? $specific_alt : $global_alt;
    
    return [
        'alt' => $alt,
        'title' => $global_title
    ];
}
