<?php
/**
 * Theme Settings ACF Fields
 *
 * @package Base_Theme
 */

/**
 * Dynamically registers ACF field groups for theme settings based on Polylang languages.
 */
function register_language_specific_theme_settings() {
	if ( ! function_exists( 'acf_add_local_field_group' ) || ! function_exists( 'pll_languages_list' ) ) {
		return;
	}

	$languages = pll_languages_list();

	if ( empty( $languages ) ) {
		return;
	}

	foreach ( $languages as $lang_slug ) {
		acf_add_local_field_group( [
			'key'      => 'group_theme_settings_' . $lang_slug,
			'title'    => 'Theme Settings (' . strtoupper( $lang_slug ) . ')',
			'fields'   => get_theme_settings_fields_for_lang( $lang_slug ),
			'location' => [
				[
					[
						'param'    => 'options_page',
						'operator' => '==',
						'value'    => 'theme-settings-' . $lang_slug,
					],
				],
			],
			'menu_order'            => 0,
			'position'              => 'normal',
			'style'                 => 'default',
			'label_placement'       => 'top',
			'instruction_placement' => 'label',
			'hide_on_screen'        => '',
			'active'                => true,
		] );
	}
}
add_action( 'acf/init', 'register_language_specific_theme_settings' );

/**
 * Generates the field sections for a specific language.
 *
 * @param string $lang_slug The language slug (e.g., 'en').
 * @return array An array of ACF fields.
 */
function get_theme_settings_fields_for_lang( $lang_slug ) {
	return [
		// Global Button Settings Tab
		[
			'key'   => 'field_global_button_settings_tab_' . $lang_slug,
			'label' => 'Global Button Settings',
			'name'  => '',
			'type'  => 'tab',
		],
		[
			'key'          => 'field_global_button_url_' . $lang_slug,
			'label'        => 'Button URL',
			'name'         => 'global_button_url_' . $lang_slug,
			'type'         => 'text',
			'instructions' => 'Enter URL for all buttons in the template',
		],

		// Header Tab
		[
			'key'   => 'field_header_tab_' . $lang_slug,
			'label' => 'Header',
			'name'  => '',
			'type'  => 'tab',
		],
		[
			'key'           => 'field_header_logo_' . $lang_slug,
			'label'         => 'Header Logo',
			'name'          => 'header_logo_' . $lang_slug,
			'type'          => 'image',
			'instructions'  => 'Upload site logo for header',
			'return_format' => 'array',
			'preview_size'  => 'medium',
			'library'       => 'all',
		],
		[
			'key'          => 'field_header_menu_' . $lang_slug,
			'label'        => 'Header Menu',
			'name'         => 'header_menu_' . $lang_slug,
			'type'         => 'repeater',
			'instructions' => 'Add menu items',
			'layout'       => 'block',
			'button_label' => 'Add Menu Item',
			'sub_fields'   => get_menu_sub_fields( $lang_slug, 'header' ),
		],

		// Footer Tab
		[
			'key'   => 'field_footer_tab_' . $lang_slug,
			'label' => 'Footer',
			'name'  => '',
			'type'  => 'tab',
		],
		[
			'key'           => 'field_footer_logo_' . $lang_slug,
			'label'         => 'Footer Logo',
			'name'          => 'footer_logo_' . $lang_slug,
			'type'          => 'image',
			'instructions'  => 'Upload site logo for footer',
			'return_format' => 'array',
			'preview_size'  => 'medium',
			'library'       => 'all',
		],
		[
			'key'          => 'field_footer_email_' . $lang_slug,
			'label'        => 'Contact Email',
			'name'         => 'footer_email_' . $lang_slug,
			'type'         => 'email',
			'instructions' => 'Email address displayed in the footer',
		],
		[
			'key'          => 'field_footer_text_' . $lang_slug,
			'label'        => 'Footer Text',
			'name'         => 'footer_text_' . $lang_slug,
			'type'         => 'textarea',
			'instructions' => 'Text content for the footer',
			'rows'         => 4,
		],
		[
			'key'          => 'field_footer_social_' . $lang_slug,
			'label'        => 'Social Networks',
			'name'         => 'social_networks_' . $lang_slug,
			'type'         => 'repeater',
			'instructions' => 'Add social network links',
			'layout'       => 'table',
			'button_label' => 'Add Social Link',
			'sub_fields'   => [
				[
					'key'      => 'field_social_name_' . $lang_slug,
					'label'    => 'Name',
					'name'     => 'name',
					'type'     => 'text',
					'required' => 1,
				],
				[
					'key'           => 'field_social_icon_' . $lang_slug,
					'label'         => 'Icon',
					'name'          => 'icon',
					'type'          => 'image',
					'return_format' => 'array',
					'preview_size'  => 'thumbnail',
				],
				[
					'key'      => 'field_social_url_' . $lang_slug,
					'label'    => 'URL',
					'name'     => 'url',
					'type'     => 'text',
					'required' => 1,
				],
			],
		],
		[
			'key'          => 'field_footer_menu_items_' . $lang_slug,
			'label'        => 'Footer Menu Items',
			'name'         => 'footer_menu_items_' . $lang_slug,
			'type'         => 'repeater',
			'instructions' => 'Add footer menu links',
			'layout'       => 'table',
			'button_label' => 'Add Footer Link',
			'sub_fields'   => [
				[
					'key'      => 'field_footer_link_title_' . $lang_slug,
					'label'    => 'Title',
					'name'     => 'title',
					'type'     => 'text',
					'required' => 1,
				],
				[
					'key'      => 'field_footer_link_url_' . $lang_slug,
					'label'    => 'URL',
					'name'     => 'url',
					'type'     => 'text',
					'required' => 1,
				],
			],
		],
		[
			'key'           => 'field_footer_images_' . $lang_slug,
			'label'         => 'Footer Images',
			'name'          => 'footer_images_' . $lang_slug,
			'type'          => 'gallery',
			'instructions'  => 'Upload images for the footer (e.g., payment systems)',
			'return_format' => 'array',
			'insert'        => 'append',
		],
	];
}

/**
 * Returns the sub-fields for a menu repeater.
 *
 * @param string $lang_slug The language slug.
 * @param string $prefix A prefix for key uniqueness.
 * @return array An array of ACF sub-fields.
 */
function get_menu_sub_fields( $lang_slug, $prefix ) {
	return [
		[
			'key'      => 'field_' . $prefix . '_menu_item_title_' . $lang_slug,
			'label'    => 'Title',
			'name'     => 'title',
			'type'     => 'text',
			'required' => 1,
		],
		[
			'key'   => 'field_' . $prefix . '_menu_item_url_' . $lang_slug,
			'label' => 'URL',
			'name'  => 'url',
			'type'  => 'text',
		],
		[
			'key'   => 'field_' . $prefix . '_menu_item_target_' . $lang_slug,
			'label' => 'Open in new window',
			'name'  => 'target',
			'type'  => 'true_false',
			'ui'    => 1,
		],
		[
			'key'          => 'field_' . $prefix . '_submenu_items_' . $lang_slug,
			'label'        => 'Submenu Items',
			'name'         => 'submenu',
			'type'         => 'repeater',
			'layout'       => 'table',
			'button_label' => 'Add Submenu Item',
			'sub_fields'   => [
				[
					'key'      => 'field_' . $prefix . '_submenu_item_title_' . $lang_slug,
					'label'    => 'Title',
					'name'     => 'title',
					'type'     => 'text',
					'required' => 1,
				],
				[
					'key'   => 'field_' . $prefix . '_submenu_item_url_' . $lang_slug,
					'label' => 'URL',
					'name'  => 'url',
					'type'  => 'text',
				],
				[
					'key'   => 'field_' . $prefix . '_submenu_item_target_' . $lang_slug,
					'label' => 'Open in new window',
					'name'  => 'target',
					'type'  => 'true_false',
					'ui'    => 1,
				],
			],
		],
	];
}