<?php
/**
 * Base Theme functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Base_Theme
 */

if (!defined('_S_VERSION')) {
    // Replace the version number of the theme on each release.
    define('_S_VERSION', '1.0.0');
}

/**
 * Include theme settings fields
 */
require get_template_directory() . '/fields/theme-settings-fields.php';
require get_template_directory() . '/fields/main-page-fields.php';
require get_template_directory() . '/fields/ex-page-fields.php';

require get_template_directory() . '/inc/template-helpers.php';

/**
 * Sets up theme defaults and registers support for various WordPress features.
 */
function base_theme_setup() {
    /*
     * Make theme available for translation.
     * Translations can be filed in the /languages/ directory.
     */
    load_theme_textdomain('base-theme', get_template_directory() . '/languages');

    // Add default posts and comments RSS feed links to head.
    add_theme_support('automatic-feed-links');

    /*
     * Let WordPress manage the document title.
     * By adding theme support, we declare that this theme does not use a
     * hard-coded <title> tag in the document head, and expect WordPress to
     * provide it for us.
     */
    add_theme_support('title-tag');

    /*
     * Enable support for Post Thumbnails on posts and pages.
     *
     * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
     */
    add_theme_support('post-thumbnails');

    // This theme uses wp_nav_menu() in one location.
    register_nav_menus(
        array(
            'menu-1' => esc_html__('Primary', 'base-theme'),
        )
    );

    /*
     * Switch default core markup for search form, comment form, and comments
     * to output valid HTML5.
     */
    add_theme_support(
        'html5',
        array(
            'search-form',
            'comment-form',
            'comment-list',
            'gallery',
            'caption',
            'style',
            'script',
        )
    );

    // Add theme support for selective refresh for widgets.
    add_theme_support('customize-selective-refresh-widgets');

    if ( function_exists( 'acf_add_options_page' ) ) {

		// Add main options page.
		$parent = acf_add_options_page( [
			'page_title' => 'Theme settings',
			'menu_title' => 'Theme settings',
			'menu_slug'  => 'theme-settings',
			'capability' => 'edit_posts',
			'redirect'   => true,
		] );

		// Add sub-pages for each language if Polylang is active.
		if ( function_exists( 'pll_languages_list' ) ) {
			$languages = pll_languages_list();
			if ( ! empty( $languages ) ) {
				foreach ( $languages as $lang_slug ) {
					acf_add_options_sub_page( [
						'page_title'  => 'Settings ' . strtoupper( $lang_slug ),
						'menu_title'  => 'Settings ' . strtoupper( $lang_slug ),
						'parent_slug' => $parent['menu_slug'], // Correct parent slug
						'menu_slug'   => 'theme-settings-' . $lang_slug,
						'post_id'     => 'options_' . $lang_slug // Unique post_id for each language
					] );
				}
			}
		} else {
			// Fallback if Polylang is not active.
			acf_add_options_sub_page( [
				'page_title'  => 'General Settings',
				'menu_title'  => 'General',
				'parent_slug' => $parent['menu_slug'],
			] );
		}
	}
}
add_action('after_setup_theme', 'base_theme_setup');

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function base_theme_content_width() {
    $GLOBALS['content_width'] = apply_filters('base_theme_content_width', 640);
}
add_action('after_setup_theme', 'base_theme_content_width', 0);

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function base_theme_widgets_init() {
    register_sidebar(
        array(
            'name'          => esc_html__('Sidebar', 'base-theme'),
            'id'            => 'sidebar-1',
            'description'   => esc_html__('Add widgets here.', 'base-theme'),
            'before_widget' => '<section id="%1$s" class="widget %2$s">',
            'after_widget'  => '</section>',
            'before_title'  => '<h2 class="widget-title">',
            'after_title'   => '</h2>',
        )
    );
}
add_action('widgets_init', 'base_theme_widgets_init');

/**
 * Enqueue scripts and styles.
 */
function base_theme_scripts() {
    wp_enqueue_style('base-theme-style', get_stylesheet_uri(), array(), _S_VERSION);
    wp_style_add_data('base-theme-style', 'rtl', 'replace');

    wp_enqueue_script('base-theme-navigation', get_template_directory_uri() . '/assets/public/js/navigation.js', array(), _S_VERSION, true);

    if (is_singular() && comments_open() && get_option('thread_comments')) {
        wp_enqueue_script('comment-reply');
    }
}
add_action('wp_enqueue_scripts', 'base_theme_scripts');

// CUSTOM
// 
// 

function custom_strings()
{
	if (function_exists('pll_register_string')) {
		pll_register_string('Log in', 'log_in');
		pll_register_string('Sign up', 'sign_up');
        pll_register_string('Get a Bonus', 'get_a_bonus');
        pll_register_string('Sign up now', 'sign_up_now');
		pll_register_string('Table of Content', 'table_of_contents');
		pll_register_string('Follow us', 'follow_us');
		pll_register_string('Need help?', 'need_help');
		pll_register_string('Base footer name', 'base_footer_name');
		pll_register_string('Language', 'language');
	}
}
add_action('after_setup_theme', 'custom_strings');
